/** @file   WarehouseFactory.cpp
 * @brief   Implementation of WarehouseFactory class.
 * @version $Revision: 1.4 $
 * @date    $Date: 2006/02/21 20:48:27 $
 * @author  Tomi Lamminsaari
 */

#include "WarehouseFactory.h"
#include <fstream>
#include "filelist.h"
#include "datatypes.h"
#include "DataWarehouse.h"
#include "BaseContainer.h"
#include "AnimContainer.h"
#include "eng2dPrivateConstants.h"
#include "framegrabber.h"
#include "StreamUtils.h"
#include "GraphicsContainer.h"
#include "spritetools.h"
#include "DatafileContainer.h"

namespace eng2d {


///
/// Constants, datatypes and static methods
/// ============================================================================

/** Constructs an animation warehouse.
 */
void WarehouseFactory::CreateWarehouse( DataWarehouse* aWarehouse,
                                        const FileList& aFiles,
                                        MWarehouseCreationObserver* aObserver )
                                        throw (xRuntimeError)
{
  for ( int i=0; i < aFiles.filecount(); i++ ) {
    int status = AppendWarehouse( aWarehouse, aFiles.getFile(i) );
    if ( status != KErrNone ) {
      std::string message = "Error ";
      message += Int2String( status );
      message += " with file: " + aFiles.getFile(i);
      throw xRuntimeError( "WarehouseFactory",
                           "CreateWarehouse(DataWarehouse, const FileList&)",
                           message );
    }
    
    if ( aObserver != 0 ) {
      aObserver->NotifyWarehouseCreation( i, aFiles.filecount(),
                                          aFiles.getFile(i) );
    }
  }
}


///
/// Protected and private methods
/// ============================================================================

int WarehouseFactory::AppendWarehouse( DataWarehouse* aWarehouse,
                                        const std::string& aFilename )
{
  std::ifstream fin( aFilename.c_str() );
  if ( !fin ) {
    return KErrNotFound;
  }
  std::string tmp;
  while ( true ) {
    if ( fin.eof() == true ) {
      fin.close();
      return KErrNone;
    }
    fin >> tmp;
    if ( tmp == "#" ) {
      fin.ignore( 4096, '\n' );
      
    } else if ( tmp == "<anim_combine_v1>" ) {
      int uid = 0;
      BaseContainer* item = ReadAnimContainer( uid, fin );
      if ( item == 0 ) {
        fin.close();
        return KErrRead;
      }
      int status = aWarehouse->AddItem( uid, item );
      fin.close();
      return status;
      
    } else if ( tmp == "<graphic_combine_v1>" ) {
      int uid = 0;
      BaseContainer* item = ReadGraphicContainer( uid, fin );
      if ( item == 0 ) {
        fin.close();
        return KErrRead;
      }
      int status = aWarehouse->AddItem( uid, item );
      fin.close();
      return status;
      
    } else {
      // Unknown element encountered. Exit
      fin.close();
      return KErrNotSupported;
    }
  }
}



BaseContainer* WarehouseFactory::ReadAnimContainer( int& aUid, std::istream& aIn )
{
  aUid = -1;
  BaseContainer* item = new AnimContainer;
  std::string tmp;
  
  while ( true ) {
    if ( aIn.eof() == true ) {
      delete item;
      return 0;
    }
    
    aIn >> tmp;
    if ( tmp == "#" ) {
      aIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "uid:" ) {
      aIn >> tmp;
      aUid = atoi( tmp.c_str() );
      
    } else if ( tmp == "<eng2d_animation>" ) {
      Animation* anim = new Animation;
      int status = anim->read( aIn );
      if ( status != KErrNone ) {
        delete item;  item = 0;
        delete anim;  anim = 0;
        return 0;
      }
      item->Push( anim );
      
    } else if ( tmp == "</anim_combine_v1>" ) {
      return item;
      
    }
  }
}


/** Reads the Graphics combine
 */
BaseContainer* WarehouseFactory::ReadGraphicContainer( int& aUid, std::istream& aIn )
{
  int ret = StreamUtils::SearchForTag( aIn, "<header>" );
  if ( ret != KErrNone ) {
    return 0;
  }
  
  GraphicContainerHeader header;
  ret = ReadGraphicContainerHeader( aIn, header );
  if ( ret != KErrNone ) {
    return 0;
  }
  
  if ( header.iInputMode == EDatafile ) {
    DATAFILE* datFile = load_datafile( header.iFilename.c_str() );
    if ( datFile == 0 ) {
      return 0;
    }
    aUid = header.iUid;
    DatafileContainer* item = new DatafileContainer;
    item->SetDatafile( datFile );
    return item;
  }
  
  // Read the source bitmap.
  BITMAP* source = load_bitmap( header.iFilename.c_str(), 0 );
  if ( source == 0 ) {
    return 0;
  }
  if ( header.iInputMode == ESingleBitmap ) {
    // The bitmap loaded above is the only object in the item.
    GraphicsContainer* item = new GraphicsContainer( header.iContentType );
    if ( header.iRotations != 0 ) {
      std::vector<BITMAP*> rotatedBitmaps =
          SpriteTools::getRotatedSprites( source, header.iRotations );
      for ( int i=0; i < rotatedBitmaps.size(); i++ ) {
        void* gfx = SpriteTools::ConvertToGraphicFormat(
                        rotatedBitmaps.at(i), header.iContentType );
        item->Push( gfx );
      }
      aUid = header.iUid;
      destroy_bitmap( source );   source = 0;
      return item;
    }
    // No rotations so we convert the graphic and exit.
    item->Push( SpriteTools::ConvertToGraphicFormat( source, header.iContentType ) );
    aUid = header.iUid;
    destroy_bitmap( source );   source = 0;
    return item;
    
  }
    
  // We should use a grid to pick parts of the graphic.
  destroy_bitmap( source );
  source = 0;
  ret = StreamUtils::SearchForTag( aIn, "<body>" );
  if ( ret != KErrNone ) {
    return 0;
  }
  std::vector<GraphicContainerGrid> graphicParts;
  ret = ReadGraphicContainerGrid( aIn, graphicParts );
  if ( ret != KErrNone ) {
    return 0;
  }
  // Create the GraphicsItem and FrameGrabber and pick the graphics from
  // grid.
  GraphicsContainer* item = new GraphicsContainer( header.iContentType );
  FrameGrabber* grabber = new FrameGrabber( header.iFilename, header.iGridSize );
  for ( int i=0; i < graphicParts.size(); i++ ) {
    int column = graphicParts.at(i).iColumn;
    int row = graphicParts.at(i).iRow;
    
    switch ( header.iContentType ) {
      case ( EBitmap ): {
        item->Push( grabber->grab( column, row ) );
        break;
      }
      case ( ERleSprite ): {
        item->Push( grabber->grabRLE( column, row ) );
        break;
      }
    }
  }
  delete grabber;   grabber = 0;
  aUid = header.iUid;
  return item;
}

/** Reads the header part of graphic combine.
 */
int WarehouseFactory::ReadGraphicContainerHeader( std::istream& aIn,
                                                  GraphicContainerHeader& aHeader )
{
  std::string tmp;
  while ( true ) {
    if ( aIn.eof() == true ) {
      return KErrEof;
    }
    
    aIn >> tmp;
    if ( tmp == "#" ) {
      aIn.ignore(KMaxLineLength, KCharNewLine);
      
    } else if ( tmp == "</header>" ) {
      if ( aHeader.iFilename.length() == 0 ) {
        return KErrGeneral;
      }
      if ( aHeader.iUid == KInvalidUid ) {
        return KErrGeneral;
      }
      if ( aHeader.iContentType == EUnspecified ) {
        return KErrGeneral;
      }
      return KErrNone;
      
    } else if ( tmp == "file:" ) {
      aIn >> aHeader.iFilename;
    } else if ( tmp == "store_as:" ) {
      aIn >> tmp;
      if ( tmp == "bitmap" ) {
        aHeader.iContentType = EBitmap;
      } else if ( tmp == "rle" ) {
        aHeader.iContentType = ERleSprite;
      }
    } else if ( tmp == "format:" ) {
      aIn >> tmp;
      if ( tmp == "single" ) {
        aHeader.iInputMode = ESingleBitmap;
      } else if ( tmp == "grid" ) {
        aHeader.iInputMode = EGrid;
      } else if ( tmp == "datafile" ) {
        aHeader.iInputMode = EDatafile;
      }
    } else if ( tmp == "id:" ) {
      aIn >> tmp;
      aHeader.iUid = atoi( tmp.c_str() );
    } else if ( tmp == "grid_x:" ) {
      aIn >> tmp;
      aHeader.iGridSize.iWidth = atoi( tmp.c_str() );
    } else if ( tmp == "grid_y:" ) {
      aIn >> tmp;
      aHeader.iGridSize.iHeight = atoi( tmp.c_str() );
      
    } else if ( tmp == "rotate:" ) {
      aIn >> tmp;
      aHeader.iRotations = atoi( tmp.c_str() );
      
    } else {
      // Unknown tag.
      return KErrCorrupt;
    }
  }
}



/** Reads the grid information.
 */
int WarehouseFactory::ReadGraphicContainerGrid( std::istream& aIn,
                            std::vector<WarehouseFactory::GraphicContainerGrid>& aGridParts )
{
  std::string tmp;
  while ( true ) {
    if ( aIn.eof() == true ) {
      return KErrEof;
    }
    aIn >> tmp;
    if ( tmp == "#" ) {
      aIn.ignore( KMaxLineLength, KCharNewLine );
      
    } else if ( tmp == "</body>" ) {
      return KErrNone;
      
    } else if ( tmp == "<g>" ) {
      std::string col;
      std::string row;
      aIn >> col >> row;
      GraphicContainerGrid grid;
      grid.iColumn = atoi( col.c_str() );
      grid.iRow = atoi( row.c_str() );
      aGridParts.push_back( grid );
      aIn >> tmp;   // read the closing tag
      
    } else {
      return KErrCorrupt;
    }
  }
}


/** GraphicItemHeader constructor.
 */
WarehouseFactory::GraphicContainerHeader::GraphicContainerHeader() :
  iContentType( EUnspecified ),
  iUid( KInvalidUid ),
  iFilename( "" ),
  iGridSize( 0,0 ),
  iInputMode( ESingleBitmap ),
  iRotations( 0 )
{
}

}

